/* Copyright (c) 2016-2018 VMware, Inc. All rights reserved. */
package com.vmware.automatic.plugin.registration.services;

import com.vmware.vim25.Description;
import com.vmware.vim25.Extension;
import com.vmware.vim25.ExtensionClientInfo;
import com.vmware.vim25.ExtensionResourceInfo;
import com.vmware.vim25.ExtensionServerInfo;
import com.vmware.vim25.KeyValue;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.MissingArgumentException;
import org.apache.commons.cli.Option;

import javax.xml.datatype.DatatypeConfigurationException;
import javax.xml.datatype.DatatypeFactory;
import javax.xml.datatype.XMLGregorianCalendar;
import java.util.GregorianCalendar;
import java.util.TimeZone;

/**
 * A service that provides public methods for update of pluginExtension's state.
 */
public class PluginExtensionRegistryService {
   /**
    * Extension types
    */
   private static final String LOCAL_PLUGIN_TYPE = "vsphere-client-serenity";
   private static final String REMOTE_PLUGIN_TYPE = "vsphere-client-remote";

   /**
    * Extension server protocols
    */
   private static final String HTTPS_PROTOCOL = "HTTPS";
   private static final String HTTP_PROTOCOL = "HTTP";

   /**
    * Extension server adminEmail
    */
   private static final String ADMIN_EMAIL = "noreply@vmware.com";

   /**
    * Updates top level properties of the given extension instance.
    *
    * @param extension - current extension's instance
    * @param commandLine - command line arguments
    */
   public void updateTopLevelProperties(final Extension extension,
         CommandLine commandLine) {
      final String version = commandLine.getOptionValue("v");
      final String company = commandLine.getOptionValue("c");
      final boolean showInSolutionManager = commandLine.hasOption("show");

      // Set the key if extension is new
      if (extension.getKey() == null) {
         extension.setKey(commandLine.getOptionValue("k"));
      }

      extension
            .setVersion((version != null) ? version : extension.getVersion());
      extension
            .setCompany(company != null ? company : extension.getCompany());
      extension.setShownInSolutionManager(showInSolutionManager);
   }

   /**
    * Updates extension's description.
    *
    * @param extension - current extension's instance
    * @param commandLine - command line arguments
    */
   public void updateDescription(final Extension extension,
         final CommandLine commandLine) {
      final String name = commandLine.getOptionValue("n");
      final String summary = commandLine.getOptionValue("s");
      final Description description;
      if (extension.getDescription() != null) {
         description = extension.getDescription();
      } else {
         description = new Description();
         description.setLabel("");
         description.setSummary("");
      }
      description.setLabel((name != null) ? name : description.getLabel());
      description
            .setSummary((summary != null) ? summary : description.getSummary());
      extension.setDescription(description);
   }

   /**
    * Updates the current extension's client info, or creates a new one,
    * if missing and any clientInfo's attributes are provided.
    *
    * @param extension - current extension's instance.
    * @param commandLine - command line arguments
    */
   public void updateClientInfo(Extension extension, CommandLine commandLine) {
      final String version = commandLine.getOptionValue("v");
      final String company = commandLine.getOptionValue("c");
      final String pluginUrl = commandLine.getOptionValue("pu");
      final boolean isRemotePlugin = commandLine.hasOption("remote");

      final ExtensionClientInfo extClientInfo;
      if (extension.getClient().size() > 0) {
         extClientInfo = extension.getClient().get(0);
      } else {
         // Create new ExtensionClientInfo instance and set the not nullables
         extClientInfo = new ExtensionClientInfo();
         extClientInfo.setType(
               isRemotePlugin ? REMOTE_PLUGIN_TYPE : LOCAL_PLUGIN_TYPE);
         extClientInfo.setCompany("");
      }
      extClientInfo.setVersion(
            (version != null) ? version : extClientInfo.getVersion());
      extClientInfo.setCompany(
            (company != null) ? company : extClientInfo.getCompany());
      extClientInfo.setDescription(extension.getDescription());
      extClientInfo
            .setUrl(pluginUrl != null ? pluginUrl : extClientInfo.getUrl());
      if (extension.getClient().size() == 0) {
         extension.getClient().add(extClientInfo);
      } else {
         extension.getClient().set(0, extClientInfo);
      }
   }

   /**
    * Updates the current extension's resource info,
    * or creates a new one if missing and extension's name attribute is provided.
    *
    * @param extension - current extension's instance
    * @param commandLine - command line arguments
    */
   public void updateResourceInfo(Extension extension, CommandLine commandLine) {
      final String name = commandLine.getOptionValue("n");
      final ExtensionResourceInfo extResourceInfo;
      if (extension.getResourceList().size() > 0) {
         extResourceInfo = extension.getResourceList().get(0);
      } else {
         // Create new ExtensionResourceInfo instance and set the not nullables
         extResourceInfo = new ExtensionResourceInfo();
         extResourceInfo.setLocale("en_US");
         extResourceInfo.setModule("name");
         final KeyValue kvPair = new KeyValue();
         kvPair.setKey("name");
         kvPair.setValue("");
         extResourceInfo.getData().add(kvPair);
      }
      extResourceInfo.getData().get(0).setValue(name != null ?
            name : extResourceInfo.getData().get(0).getValue());
      if (extension.getResourceList().size() > 0) {
         extension.getResourceList().set(0, extResourceInfo);
      } else {
         extension.getResourceList().add(extResourceInfo);
      }
   }

   /**
    * Updates the current extension's server info, or creates a new one,
    * if missing and protocol is HTTPS.
    * If protocol is HTTP - no server info is required.
    *
    * @param extension - the current extension's instance
    * @param commandLine - command line arguments
    */
   public void updateServerInfo(Extension extension, CommandLine commandLine) throws
         MissingArgumentException {
      // HTTPS requests require server info.
      // Update server info IF the new pluginUrl (if provided) is HTTPS,
      // or the current ServerInfo type is HTTPS AND there is at least one
      // ServerInfo property provided for update.
      // Else - No Server Info required.
      final String pluginUrl = commandLine.getOptionValue("pu");
      final String serverThumbprint = commandLine.getOptionValue("st");
      final String company = commandLine.getOptionValue("c");
      ExtensionServerInfo extServerInfo;
      if (!isNewPluginUrlHttp(pluginUrl) && isCurrentExtensionHttps(extension)
            && anyServerInfoMemberToUpdate(commandLine) || isNewPluginUrlHttps(
            pluginUrl)) {
         if (extension.getServer().size() > 0) {
            extServerInfo = extension.getServer().get(0);
         } else if (serverThumbprint == null) {
            throw new MissingArgumentException(Option.builder("st").build());
         } else {
            extServerInfo = new ExtensionServerInfo();
            extServerInfo.setType(HTTPS_PROTOCOL);
            extServerInfo.getAdminEmail().add(ADMIN_EMAIL);
            extServerInfo.setCompany("");
         }
         extServerInfo.setDescription(extension.getDescription());
         extServerInfo.setServerThumbprint(serverThumbprint != null ?
               serverThumbprint :
               extServerInfo.getServerThumbprint());
         extServerInfo.setUrl(
               pluginUrl != null ? pluginUrl : extServerInfo.getUrl());
         extServerInfo.setCompany(
               company != null ? company : extServerInfo.getCompany());
         if (extension.getServer().size() == 0) {
            extension.getServer().add(extServerInfo);
         } else if (isNewPluginUrlHttp(pluginUrl)) {
            extension.getServer().set(0, extServerInfo);
         }
      } else if (isNewPluginUrlHttp(pluginUrl)){
         System.out.println(
               "INFO: Not using https for your plugin URL is OK for testing but not recommended for production."
                     + "\nUsers will have to include the flag allowHttp=true in their vSphere Client webclient.properties otherwise the http URL will be ignored");
         extension.getServer().clear();
      }
   }

   /**
    * Updates last modification time of a given extension.
    *
    * @param extension - the current extension's instance
    * @throws DatatypeConfigurationException
    */
   public void updatelastHeartbeatTime(Extension extension)
         throws DatatypeConfigurationException {
      GregorianCalendar cal = new GregorianCalendar(
            TimeZone.getTimeZone("GMT"));
      DatatypeFactory dtFactory = DatatypeFactory.newInstance();
      XMLGregorianCalendar xmlCalendar = dtFactory.newXMLGregorianCalendar(cal);
      extension.setLastHeartbeatTime(xmlCalendar);
   }

   // Private helper methods
   /**
    * Checks if the new pluginUrl, or the existing extServerInfo is of type HTTPS
    * @param extension - the current extension instance.
    */
   private boolean isCurrentExtensionHttps(Extension extension) {
      return extension.getServer().size() > 0 && HTTPS_PROTOCOL
            .equalsIgnoreCase(extension.getServer().get(0).getType());
   }

   /**
    * Checks if the provided pluginUrl parameter is HTTPS.
    */
   private boolean isNewPluginUrlHttps(String pluginUrl) {
      return pluginUrl != null && pluginUrl.toLowerCase()
            .startsWith(HTTPS_PROTOCOL.toLowerCase());
   }

   /**
    * Checks if the provided pluginUrl parameter is HTTP.
    */
   private boolean isNewPluginUrlHttp(String pluginUrl) {
      return pluginUrl != null && pluginUrl.toLowerCase()
            .startsWith(HTTP_PROTOCOL.toLowerCase().concat("://"));
   }

   /**
    * Checks if there is at least one ServerInfo property provided for update.
    */
   private boolean anyServerInfoMemberToUpdate(CommandLine commandLine) {
      return commandLine.getOptionValue("c") != null
            || commandLine.getOptionValue("st") != null
            || commandLine.getOptionValue("n") != null
            || commandLine.getOptionValue("s") != null;
   }
}
